%% 
% This is an example only.
% In this example, the script is parsing:
%   Example ground truth dynamic data
%           & 
%   Example dynamic data of the system to be evaluated: in this case, ZeroKey position data
% and computing the error between them.
% This script has a function that requires image processing and computer
% vision toolboxes. If they are not available, use Matlab online

close all
clear

%% Settings
DEBUG = false; % Display additional graphs and other intermediate information

% switch \ to become / if using Matlab Online
ZeroKeyDataFile = "TestData\zk_dynamic_example.txt";
GTDataFile = "TestData\gt_dynamic_example.txt";

ZK_Trim_Epochs = [0, 0];
GT_Trim_Epochs = [0, 0];

%explicitely list the ZeroKey flags to be included in this analysis
ZK_ins_flags_include = [4]; %to include all: [0:6], to include trusted only: [4]
ZK_us_flags_include = [3]; % to include all: [0:3], to include trusted only: [3]

GTUpsampleRate = 2e3; % Rate to upsample GT data to
GTSampleTime = 1/GTUpsampleRate;

NumTransformPoints = 16; % Number of points to use for the similarity transform
TransformIterations = 3; % Number of times to resolve transformation matrix after adjusting time offsets

%% Load data from log files
[zk_nuc_unix, zk_us_seq, zk_ins_seq, zk_pos_x,zk_pos_y,zk_pos_z, ...
 zk_quat_w,zk_quat_x,zk_quat_y,zk_quat_z,zk_vel_x,zk_vel_y,zk_vel_z,...
 zk_ins_flag,zk_us_flag,zk_relative_time,zk_tOffset] = ...
        ParseZeroKeyDataFile(ZeroKeyDataFile);

% Trim ZeroKey dataset by specified number of epochs
zk_nuc_unix = zk_nuc_unix(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_us_seq = zk_us_seq(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_ins_seq = zk_ins_seq(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_pos_x = zk_pos_x(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_pos_y = zk_pos_y(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_pos_z = zk_pos_z(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_quat_w = zk_quat_w(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_quat_x = zk_quat_x(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_quat_y = zk_quat_y(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_quat_z = zk_quat_z(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_vel_x = zk_vel_x(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_vel_y = zk_vel_y(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_vel_z = zk_vel_z(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_ins_flag = zk_ins_flag(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_us_flag = zk_us_flag(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_relative_time = zk_relative_time(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));
zk_tOffset = zk_tOffset(ZK_Trim_Epochs(1)+1:end-ZK_Trim_Epochs(2));

% filter ZK data for position flags
zk_accepted_ins_flags_idx = [];
for ins_flag_idx = 1:length(ZK_ins_flags_include)
    if isempty(zk_accepted_ins_flags_idx)
        zk_accepted_ins_flags_idx = zk_ins_flag == ZK_ins_flags_include(ins_flag_idx);
    else
        zk_accepted_ins_flags_idx = zk_accepted_ins_flags_idx | (zk_ins_flag == ZK_ins_flags_include(ins_flag_idx));
    end
end
zk_accepted_us_flags_idx = [];
for us_flag_idx = 1:length(ZK_us_flags_include)
    if isempty(zk_accepted_us_flags_idx)
        zk_accepted_us_flags_idx = zk_us_flag == ZK_us_flags_include(us_flag_idx);
    else
        zk_accepted_us_flags_idx = zk_accepted_us_flags_idx | (zk_us_flag == ZK_us_flags_include(us_flag_idx));
    end
end
zk_accepted_flags_idx = zk_accepted_ins_flags_idx & zk_accepted_us_flags_idx;

if sum(zk_accepted_flags_idx) == 0
    disp("The selected INS & US flag combinations are not present in this data. The data will be processed without flag filtering!!");
else
    zk_nuc_unix = zk_nuc_unix(zk_accepted_flags_idx);
    zk_us_seq = zk_us_seq(zk_accepted_flags_idx);
    zk_ins_seq = zk_ins_seq(zk_accepted_flags_idx);
    zk_pos_x = zk_pos_x(zk_accepted_flags_idx);
    zk_pos_y = zk_pos_y(zk_accepted_flags_idx);
    zk_pos_z = zk_pos_z(zk_accepted_flags_idx);
    zk_quat_w = zk_quat_w(zk_accepted_flags_idx);
    zk_quat_x = zk_quat_x(zk_accepted_flags_idx);
    zk_quat_y = zk_quat_y(zk_accepted_flags_idx);
    zk_quat_z = zk_quat_z(zk_accepted_flags_idx);
    zk_vel_x = zk_vel_x(zk_accepted_flags_idx);
    zk_vel_y = zk_vel_y(zk_accepted_flags_idx);
    zk_vel_z = zk_vel_z(zk_accepted_flags_idx);
    zk_ins_flag = zk_ins_flag(zk_accepted_flags_idx);
    zk_us_flag = zk_us_flag(zk_accepted_flags_idx);
    zk_relative_time = zk_relative_time(zk_accepted_flags_idx);
    zk_tOffset = zk_tOffset(zk_accepted_flags_idx);
end

% Remove tOffset from ZK timestamp to get true measurement time
zk_relative_time = zk_relative_time - zk_tOffset;
% Convert ZeroKey time base to seconds and set T0 = 0
zk_relative_time = (zk_relative_time - zk_relative_time(1)) / 1e6;

[gt_nuc_unix, gt_meas_time, gt_x, gt_y, gt_z] = ParseGTDataFile(GTDataFile);

% Trim GT dataset by specified number of epochs
gt_nuc_unix = gt_nuc_unix(GT_Trim_Epochs(1)+1:end-GT_Trim_Epochs(2));
gt_meas_time = gt_meas_time(GT_Trim_Epochs(1)+1:end-GT_Trim_Epochs(2));
gt_x = gt_x(GT_Trim_Epochs(1)+1:end-GT_Trim_Epochs(2));
gt_y = gt_y(GT_Trim_Epochs(1)+1:end-GT_Trim_Epochs(2));
gt_z = gt_z(GT_Trim_Epochs(1)+1:end-GT_Trim_Epochs(2));

% Remove duplicate data lines from GT data
[gt_meas_time, IA, ~] = unique(gt_meas_time);
gt_nuc_unix = gt_nuc_unix(IA);
gt_x = gt_x(IA);
gt_y = gt_y(IA);
gt_z = gt_z(IA);

% Convert GT time base to seconds and set T0 = 0
gt_meas_time = (gt_meas_time - gt_meas_time(1)) / 1e3;

if DEBUG
    figure;
    plot3(zk_pos_x,zk_pos_y,zk_pos_z);
    title('ZeroKey Raw Position Data');
    figure;
    plot3(gt_x, gt_y, gt_z);
    title('GT Raw Position Data');
end

%% Pick some "close" points using system time for first solve of similarity transform

% Get NumTransformPoints points from the middle half of the dataset
n = length(zk_pos_x);
nS = floor(n / 2 / NumTransformPoints); % Determine step to equally space 16 samples acorss the middle 50% of the dataset
zk_stInd = floor(n/4):nS:(nS*(NumTransformPoints-1)+floor(n/4));
gt_stInd = [];

% Find closest corresponding index by system time in GT dataset
for i = 1:NumTransformPoints
    [~, gt_stInd(i)] = min(abs(gt_nuc_unix - zk_nuc_unix(zk_stInd(i))));
end

gt_pos_static_all = [gt_x(gt_stInd), gt_y(gt_stInd), gt_z(gt_stInd)];
zk_pos_static_all = [zk_pos_x(zk_stInd), zk_pos_y(zk_stInd), zk_pos_z(zk_stInd)];

%% Interpolate GT data

% Initialize arrays
int_t = gt_meas_time(1):GTSampleTime:gt_meas_time(end);
int_gt_x = zeros(size(int_t,2),1);
int_gt_y = zeros(size(int_t,2),1);
int_gt_z = zeros(size(int_t,2),1);
int_gt_x(1) = gt_x(1);
int_gt_y(1) = gt_y(1);
int_gt_z(1) = gt_z(1);

% Compute interpolated positions based on variable measurement interval
% from GT data and velocity between measured points
for i = 2:length(int_gt_x)
    % Find next sample that is later in time than current
    next_ind = find(gt_meas_time >= int_t(i),1);

    % Calculate velocity using next position and last position
    sl_x = (gt_x(next_ind) - gt_x(next_ind-1))/(gt_meas_time(next_ind)-gt_meas_time(next_ind-1));
    sl_y = (gt_y(next_ind) - gt_y(next_ind-1))/(gt_meas_time(next_ind)-gt_meas_time(next_ind-1));
    sl_z = (gt_z(next_ind) - gt_z(next_ind-1))/(gt_meas_time(next_ind)-gt_meas_time(next_ind-1));

    % Calculate position for this interpolated epoch based on velocity and
    % last position
    int_gt_x(i) = gt_x(next_ind - 1) + sl_x * (int_t(i) - gt_meas_time(next_ind - 1));
    int_gt_y(i) = gt_y(next_ind - 1) + sl_y * (int_t(i) - gt_meas_time(next_ind - 1));
    int_gt_z(i) = gt_z(next_ind - 1) + sl_z * (int_t(i) - gt_meas_time(next_ind - 1));
end


if DEBUG
    % Plot original x coordinate and interpolated x coordinate over time
    figure;
    plot(gt_meas_time,gt_x,'x');
    hold on;
    plot(int_t,int_gt_x,'.');
    title(sprintf('GT raw x-coordinate and interpolated x-coordinate (%d Hz)', GTUpsampleRate));
end

%% Iteratively solve similarity transform

for (tf_i = 1:TransformIterations)

    %% Compute similarity transform
    % Requies image processing and computer vision toolboxes
    % Run through Matlab online instead if you don't have it!
    tform = estgeotform3d(gt_pos_static_all,zk_pos_static_all,"similarity",'MaxNumTrials',1e5, 'Confidence',99.9999);
    
    tf_gt_pos_static = [gt_pos_static_all ones(size(gt_pos_static_all,1),1)];
    
    tf_gt_pos_static = tf_gt_pos_static*tform.A';
    tf_gt_pos_static = tf_gt_pos_static(:,1:3);
    
    if DEBUG
        % Plot static points of both systems after transformation
        figure;
        plot3(zk_pos_static_all(:,1),zk_pos_static_all(:,2),zk_pos_static_all(:,3),'*');
        hold on
        plot3(tf_gt_pos_static(:,1),tf_gt_pos_static(:,2),tf_gt_pos_static(:,3),'sq');
        title('Transformed GT control points over ZK points');
    end
    
    fullTran = tform.A;
    
    staticRMSE = sqrt(mean(sum(((zk_pos_static_all - tf_gt_pos_static).^2)')));
    
    disp(sprintf('RMSE of points used for transformation matrix is (itr %d): %.5f', tf_i, staticRMSE));
    
    %% Transform GT data to ZK coordinate system
    int_pos_GT = [int_gt_x int_gt_y int_gt_z ones(length(int_gt_x),1)];
    
    int_pos_GT = int_pos_GT*fullTran';
    int_pos_GT = int_pos_GT(:,1:3);
    
    int_gt_x_tran = int_pos_GT(:,1);
    int_gt_y_tran = int_pos_GT(:,2);
    int_gt_z_tran = int_pos_GT(:,3);
    
    %% Find indicies of closest in time upsampled GT epochs corresponding to ZK epochs
    corr_int_gt_idx = nan(size(zk_relative_time,1),1);
    
    % For each ZK epoch, find the closest in time upsampled GT epoch 
    for i = 1:length(zk_relative_time)
        % Find the closest (in time) sample of the GT data to the
        % current ZK sample
        [~,corr_ind] = min(abs(int_t - zk_relative_time(i)));

        % Skip indexes that are the same as the last (one dataset is
        % longer than the other)
        if ((i > 1) && ((corr_int_gt_idx(i-1) == corr_ind) || isnan(corr_int_gt_idx(i-1))))
            continue;
        end
        corr_int_gt_idx(i) = corr_ind;
    end

    % Remove nan elements
    corr_int_gt_idx = corr_int_gt_idx(~isnan(corr_int_gt_idx));

    %% Search for best time alignment of datasets

    % Search through all time shifts from -1/4 of all GT data to +1/4 of GT
    % data
    rmse = nan(ceil(length(int_gt_x_tran)/2),1);
    n_gt = length(int_gt_x_tran);
    
    parfor shiftI = 1:ceil(length(int_gt_x_tran)/2)
        shift = shiftI - floor(length(int_gt_x_tran)/4);
        shifted_corr = corr_int_gt_idx + shift;
        min_valid_ind = find(shifted_corr >= 1, 1);
        max_valid_ind = find(shifted_corr > n_gt, 1);

        if (isempty(min_valid_ind))
            min_valid_ind = 1;
        end
        if (isempty(max_valid_ind))
            max_valid_ind = length(shifted_corr);
        else
            max_valid_ind = max_valid_ind - 1;
        end

        error_x = int_gt_x_tran(shifted_corr(min_valid_ind:max_valid_ind)) - zk_pos_x(min_valid_ind:max_valid_ind);
        error_y = int_gt_y_tran(shifted_corr(min_valid_ind:max_valid_ind)) - zk_pos_y(min_valid_ind:max_valid_ind);
        error_z = int_gt_z_tran(shifted_corr(min_valid_ind:max_valid_ind)) - zk_pos_z(min_valid_ind:max_valid_ind);

        rmse(shiftI) = sqrt(mean(error_x .^ 2 + error_y .^ 2 + error_z .^ 2));
    end
    
    % Determine the best time match based on lowest RMSE
    [~, toI] = min(rmse);
    shift = toI - floor(length(int_gt_x_tran)/4);
    
    % Recompute the proper matching indicies
    shifted_corr = corr_int_gt_idx + shift;
    valid_ind = find(shifted_corr >= 1 & shifted_corr <= length(int_gt_x_tran));
    
    time_offset = shift * GTSampleTime;
    disp(sprintf('Best time offset calculated to be: %.5f (RMSE %.5f)', time_offset, min(rmse)));
    
    if DEBUG
        figure;
        plot(rmse);
        title(sprintf('RMSE over time shift 1/%d', GTUpsampleRate));

        % Calculate magnitude of errors
        mGtZk = sqrt( ...
           ((int_gt_x_tran(shifted_corr(valid_ind)) - zk_pos_x(valid_ind)) .^ 2) + ...
           ((int_gt_y_tran(shifted_corr(valid_ind)) - zk_pos_y(valid_ind)) .^ 2) + ...
           ((int_gt_z_tran(shifted_corr(valid_ind)) - zk_pos_z(valid_ind)) .^ 2) );

        minMGtZk = min(mGtZk);
        maxMGtZk = max(mGtZk);

        figure;
        for i = 1:length(valid_ind)
            hold on;
            plot3([int_gt_x_tran(shifted_corr(valid_ind(i))) zk_pos_x(valid_ind(i))],...
                [int_gt_y_tran(shifted_corr(valid_ind(i))) zk_pos_y(valid_ind(i))],...
                [int_gt_z_tran(shifted_corr(valid_ind(i))) zk_pos_z(valid_ind(i))],'k');

            pctError = (mGtZk(i) - minMGtZk) / (maxMGtZk-minMGtZk);
            plot3(zk_pos_x(valid_ind(i)),zk_pos_y(valid_ind(i)),zk_pos_z(valid_ind(i)),'ro', 'LineWidth', 1, 'Color', hsv2rgb([(1-pctError) * 0.15,0.9,0.9]));
        end
        title('Mapping between GT and corresponding ZK points');
    end

%% Continue iteration loop for transform solving using time shifted pos data

    gt_pos_static_all = [int_gt_x(shifted_corr(valid_ind)), int_gt_y(shifted_corr(valid_ind)), int_gt_z(shifted_corr(valid_ind))];
    zk_pos_static_all = [zk_pos_x(valid_ind), zk_pos_y(valid_ind), zk_pos_z(valid_ind)];
end


%% Calculate RMSE and display graphs

error_x = int_gt_x_tran(shifted_corr(valid_ind)) - zk_pos_x(valid_ind);
error_y = int_gt_y_tran(shifted_corr(valid_ind)) - zk_pos_y(valid_ind);
error_z = int_gt_z_tran(shifted_corr(valid_ind)) - zk_pos_z(valid_ind);

rmse_xyz = sqrt(mean(error_x.^2 + error_y.^2 + error_z.^2));
rmse_xy = sqrt(mean(error_x.^2 + error_y.^2));
error_mag = sqrt(error_x.^2+error_y.^2+error_z.^2);

T = table(rmse_xy, rmse_xyz, mean(error_mag), median(error_mag), std(error_mag), min(error_mag), max(error_mag), 'VariableNames', {'RMSE (XY)','RMSE (XYZ)','Mean','Median','STD','Min','Max'})

% Plot final tranjectories
figure;
plot(sqrt(error_x.^2+error_y.^2+error_z.^2));
title('Error by epoch');
figure;
plot3(zk_pos_x(valid_ind),zk_pos_y(valid_ind),zk_pos_z(valid_ind),'b.')
hold on;
plot3(int_gt_x_tran(shifted_corr(valid_ind)), int_gt_y_tran(shifted_corr(valid_ind)), int_gt_z_tran(shifted_corr(valid_ind)),'r.')
legend("ZeroKey","GT");
title('Final transformed plots of GT and ZeroKey points');

disp(sprintf('RMSE for XYZ is %.5f', rmse_xyz));
disp(sprintf('RMSE for XY is %.5f', rmse_xy));


%% Parsing functions
% Parses a ZeroKey log file output by client's tool
function [zk_nuc_unix,zk_us_seq,zk_ins_seq,zk_pos_x,zk_pos_y,zk_pos_z,zk_quat_w,zk_quat_x,zk_quat_y,zk_quat_z,zk_vel_x,zk_vel_y,zk_vel_z,zk_ins_flag,zk_us_flag,zk_relative_time,zk_tOffset] = ParseZeroKeyDataFile(filename)
    
    zk_nuc_unix = [];
    zk_us_seq = [];
    zk_ins_seq = [];
    zk_pos_x = [];
    zk_pos_y = [];
    zk_pos_z = [];
    zk_quat_w = [];
    zk_quat_x = [];
    zk_quat_y = [];
    zk_quat_z = [];
    zk_vel_x = [];
    zk_vel_y = [];
    zk_vel_z = [];
    zk_ins_flag = [];
    zk_us_flag = [];
    zk_relative_time = [];
    zk_tOffset = [];

    zk_fid = fopen(filename);

    line = fgetl(zk_fid);
    while (line ~= -1)
        line = string(line);
        if ~contains(line, "position")
            continue
        end
        %NUC_UNIX: 1716610767.9475427 : {"header": {"stamp": {"sec": 1716581386, "nanosec": 410900115}}, "unix_time": 1716581386.4109, "relative_time": "unavailable", "device_id": "F4:AC:E9:BF:A1:9F", "flag": "00", "UsSeq": 36093, "InsSeq": 4, "position": {"x": 1.31763, "y": 12.44981, "z": 0.44097}, "orientation": {"w": 0.73655, "x": 0.01408, "y": 0.0084, "z": -0.67619}, "velocity": {"x": 0.00322, "y": 0.0001, "z": 0.0005}, "pf": "43", "ts": "4962378512", "toffset": "16590"}
        pVals = regexp(line, '^.*NUC_UNIX: (?<NucUnix>[\d.]+) .*"UsSeq": (?<UsSeq>\d+),.*"InsSeq": (?<InsSeq>\d+),.*"position": {"x": (?<x>[-\d.Ee]+), "y": (?<y>[-\d.Ee]+), "z": (?<z>[-\d.Ee]+)}.*"orientation": {"w": (?<qw>[-\d.Ee]+), "x": (?<qx>[-\d.Ee]+), "y": (?<qy>[-\d.Ee]+), "z": (?<qz>[-\d.Ee]+)}.*"velocity": {"x": (?<vx>[-\d.Ee]+), "y": (?<vy>[-\d.Ee]+), "z": (?<vz>[-\d.Ee]+)}.*"pf": "(?<pf>[^"]*)".*"ts": "(?<ts>\d+)".*"toffset": "(?<toffset>\d+)"', 'names');
        if isempty(pVals)
            error(strcat('Error: Could not parse line:', line));
        end

        zk_nuc_unix = [zk_nuc_unix; str2double(pVals.NucUnix)];
        zk_us_seq = [zk_us_seq; str2double(pVals.UsSeq)];
        zk_ins_seq = [zk_ins_seq; str2double(pVals.InsSeq)];
        zk_pos_x = [zk_pos_x; str2double(pVals.x)];
        zk_pos_y = [zk_pos_y; str2double(pVals.y)];
        zk_pos_z = [zk_pos_z; str2double(pVals.z)];

        zk_quat_w = [zk_quat_w; str2double(pVals.qw)];
        zk_quat_x = [zk_quat_x; str2double(pVals.qx)];
        zk_quat_y = [zk_quat_y; str2double(pVals.qy)];
        zk_quat_z = [zk_quat_z; str2double(pVals.qz)];

        zk_vel_x = [zk_vel_x; str2double(pVals.vx)];
        zk_vel_y = [zk_vel_y; str2double(pVals.vy)];
        zk_vel_z = [zk_vel_z; str2double(pVals.vz)];

        zk_ins_flag = [zk_ins_flag; str2double(pVals.pf{1}(1))];
        zk_us_flag = [zk_us_flag; str2double(pVals.pf{1}(2))];
        zk_relative_time = [zk_relative_time; str2double(pVals.ts)];
        zk_tOffset = [zk_tOffset; str2double(pVals.toffset)];

        line = fgetl(zk_fid);
    end
    fclose(zk_fid);
end

% Parses a GT log file
function [gt_nuc_unix, gt_meas_time, gt_x, gt_y, gt_z] = ParseGTDataFile(filename)
    gt_nuc_unix = [];
    gt_meas_time = [];
    gt_x = [];
    gt_y = [];
    gt_z = [];
    
    gt_fid = fopen(filename);
    gt_meas_time = [];
    line = fgetl(gt_fid);
    while(line ~= -1)
        line = string(line);
        if ~isempty(line)
            str_split = strsplit(line);
            gt_nuc_unix = [gt_nuc_unix; str2double(str_split(2))];
            gt_meas_time = [gt_meas_time; str2double(str_split(4))];
            gt_x = [gt_x; str2double(str_split(6))];
            gt_y = [gt_y; str2double(str_split(7))];
            gt_z = [gt_z; str2double(str_split(8))];
        end
        line = fgetl(gt_fid);
    end
    fclose(gt_fid);
end